/************************************************************************/
/*                                                                      */
/* Borland Enterprise Core Objects                                      */
/*                                                                      */
/* Copyright (c) 2003-2005 Borland Software Corporation                 */
/*                                                                      */
/************************************************************************/

using System;
using System.Collections;
using System.ComponentModel;
using System.Windows.Forms;
using System.Reflection;
using System.Drawing; // For ToolboxBitmap

using Borland.Eco.Handles;
using Borland.Eco.ObjectRepresentation;
using Borland.Eco.AutoContainers;
using Borland.Eco.Globalization;

namespace Borland.Eco.WinForm
{
	///<summary>
	///This class extends grids and listboxes with new properties; EcoAutoForm and EcoAutoFormReadOnly.
	///The purpose is to allow double-click on the extended components to bring up an ECO autocontainer
	///without having to go through the hassle of writing code.
	///</summary>
	[ProvideProperty("EcoAutoForm", typeof(Control))]
	[ProvideProperty("EcoAutoFormReadOnly", typeof(Control))]
	[ToolboxBitmap(typeof(EcoAutoFormExtender), "Borland.Eco.WinForm.EcoAutoFormExtender.bmp")]
	[ToolboxItem(true)]
	[ToolboxItemFilter("System.Windows.Forms")]
	public class EcoAutoFormExtender: System.ComponentModel.Component, System.ComponentModel.IExtenderProvider
	{
		private Hashtable controlledComponents = new Hashtable();
		private Hashtable readOnlyComponents = new Hashtable();

		#region IExtenderProvider implementation
		///<summary>
		///This method determines which controls are to be considered extensible, i.e. receive new properties.
		///You may override this method in a subclass if you need additional (or other) rules to determine
		///which controls to extend.
		///</summary>		
		public virtual bool CanExtend(object extendee)
		{
			return ((extendee is ListBox) || (extendee is DataGrid)); // only hook to certain types
		}
		#endregion

		#region Provided EcoAutoForm property
		/// <summary>
		/// Determines if the extendee should bring up an autoform on doubleclick.
		/// </summary>
		///<param name="control">Control to query.</param>
		[LocalizableCategory(typeof(EcoAutoFormExtender), "sCategoryEcoGui")]
		[LocalizableDescription(typeof(EcoAutoFormExtender), "sPropertyEcoAutoForm")]
		[DefaultValue(false)]
		public bool GetEcoAutoForm(Control control)
		{
			return controlledComponents[control] == null ? false : (bool)controlledComponents[control];
		}
		///<summary>
		///This virtual method hooks up the extended control's event. You may want some other event hooked, or hook to something else.
		///</summary>
		///<exception cref="ArgumentNullException">Thrown if <paramref name="control"/> is null.</exception>
		protected virtual void Hookup(Control control)
		{
			if (control == null) throw new ArgumentNullException("control"); // Do not localize
			control.DoubleClick += new EventHandler(OnDoubleClick);
			control.MouseDown += new MouseEventHandler(OnMouseDown);
		}
		///<summary>
		///This virtual method unhooks up the extended controls event.
		///</summary>
		///<exception cref="ArgumentNullException">Thrown if <paramref name="control"/> is null.</exception>
		protected virtual void Unhook(Control control)
		{
			if (control == null) throw new ArgumentNullException("control"); // Do not localize
			control.DoubleClick -= new EventHandler(OnDoubleClick);
			control.MouseDown -= new MouseEventHandler(OnMouseDown);
		}
		
		///<summary>
		///Setter for EcoAutoForm.
		///</summary>
		///<exception cref="ArgumentNullException">Thrown if <paramref name="control"/> is null.</exception>
		///<param name="control">Control to extend.</param>
		///<param name="value">Determines if the <paramref name="control"/> should bring up an autoform on doubleclick.</param>		
		public void SetEcoAutoForm(Control control, bool value)
		{
			if (control == null) throw new ArgumentNullException("control"); // Do not localize
			if (!value)
			{
				controlledComponents.Remove(control);
				Unhook(control);
			}
			else
			{
				controlledComponents[control] = value;
				Hookup(control);
			}
		}

		/// <summary>
		/// Determines if the autoform should be read only.
		/// </summary>
		///<param name="control">Control to query.</param>
		[LocalizableCategory(typeof(EcoAutoFormExtender), "sCategoryEcoGui")]
		[LocalizableDescription(typeof(EcoAutoFormExtender), "sPropertyEcoAutoFormReadOnly")]
		[DefaultValue(false)]
		public bool GetEcoAutoFormReadOnly(Control control)
		{
			return readOnlyComponents[control] == null ? false : (bool)readOnlyComponents[control];
		}
		///<summary>
		///Setter for EcoAutoFormReadOnly. Determines if the autoform should be read only.
		///</summary>
		///<param name="control">Control to extend.</param>
		///<param name="value">Determines if the autoform is readonly.</param>		
		///<exception cref="ArgumentNullException">Thrown if <paramref name="control"/> is null.</exception>
		public void SetEcoAutoFormReadOnly(Control control, bool value)
		{
			if (!value)
				readOnlyComponents.Remove(control);
			else
				readOnlyComponents[control] = value;
		}
		#endregion

		#region Enabled property
		private bool m_Enabled = true;
		/// <summary>
		/// Determines if the extdner is active or not. Default is true.
		/// </summary>
		[Browsable(true)]
		[DefaultValue(true)]
		[LocalizableCategory(typeof(EcoAutoFormExtender), "sCategoryBehaviour")]
		[LocalizableDescription(typeof(EcoAutoFormExtender), "sPropertyExtenderEnabled")]
		public bool Enabled
		{
			get { return m_Enabled; }
			set { m_Enabled = value; }
		}
		private bool EffectiveEnabled()
		{
			return Enabled;
		}
		#endregion
		private AutoContainerMemberVisibility m_MemberVisibility = AutoContainerMemberVisibility.AllMembers;
		/// <summary>
		/// Determines what visibility of members will be shown on the autoform. Default is <see cref="AutoContainerMemberVisibility.AllMembers"/>.
		/// </summary>
		[Browsable(true)]
		[DefaultValue(AutoContainerMemberVisibility.AllMembers)]
		[LocalizableCategory(typeof(EcoAutoFormExtender), "sCategoryBehaviour")]
		[LocalizableDescription(typeof(EcoAutoFormExtender), "sPropertyExtenderMemberVisibility")]
		public AutoContainerMemberVisibility MemberVisibility
		{
			get { return m_MemberVisibility; }
			set { m_MemberVisibility = value; }
		}
		
		private ContainerReusage m_ContainerReusage = ContainerReusage.NeverReuse;
		/// <summary>
		/// Determines how the autocontainers will be reused on a second request. Default is ContainerReusage.NeverReuse.
		/// </summary>
		[Browsable(true)]
		[DefaultValue(ContainerReusage.NeverReuse)]
		[LocalizableCategory(typeof(EcoAutoFormExtender), "sCategoryBehaviour")]
		[LocalizableDescription(typeof(EcoAutoFormExtender), "sPropertyExtenderContainerReusage")]
		public ContainerReusage ContainerReusage
		{
			get { return m_ContainerReusage; }
			set { m_ContainerReusage = value; }
		}
		private bool m_ShowModal;
		/// <summary>
		/// If true, the autocontainer will show modally if the container supports it (i.e. is a form)
		/// </summary>
		[Browsable(true)]
		[DefaultValue(false)]
		[LocalizableCategory(typeof(EcoAutoFormExtender), "sCategoryBehaviour")]
		[LocalizableDescription(typeof(EcoAutoFormExtender), "sPropertyExtenderShowModal")]
		public bool ShowModal
		{
			get { return m_ShowModal; }
			set { m_ShowModal = value; }
		}
		#region Hookup code
		bool m_LegalMousePosition = true;
		///<exception cref="ArgumentNullException">Thrown if <paramref name="e"/> is null.</exception>
		protected void OnMouseDown(object sender, MouseEventArgs e)
		{
			if (e == null) throw new ArgumentNullException("e");
			m_LegalMousePosition = true;
			DataGrid dataGrid = sender as DataGrid;
			if (dataGrid != null)
			{
				System.Windows.Forms.DataGrid.HitTestInfo myHitTest = dataGrid.HitTest(e.X, e.Y);
				m_LegalMousePosition = (myHitTest.Type == System.Windows.Forms.DataGrid.HitTestType.Cell)
															|| (myHitTest.Type == System.Windows.Forms.DataGrid.HitTestType.RowHeader);
			}
		}
		
		protected EcoSpace EffectiveEcoSpace(object o)
		{
			EcoSpace result = EcoSpace;
			if (result == null)
			{
				ElementHandle elementHandle = EcoWinFormHelper.ConnectedHandle(o);
				if (elementHandle != null)
					result = EcoWinFormHelper.ConnectedEcoSpace(elementHandle);
			}
			return result;
		}
		
		protected void ShowAutoContainer(EcoSpace ecoSpace, IElement element, bool forceReadonly)
		{
			if (element != null)
			{
				AutoContainerArgs autoContainerArgs = new AutoContainerArgs(ecoSpace, forceReadonly, MemberVisibility, ContainerReusage, ShowModal);
				IAutoContainer a = AutoContainerService.Instance.RetrieveContainer(element, autoContainerArgs);
				a.Show(autoContainerArgs);
			}
		}

		protected void OnDoubleClick(object sender, EventArgs e)
		{
			// Leave if not enabled
			if (!EffectiveEnabled() || !m_LegalMousePosition)
				return;

			if (sender == null) throw new ArgumentNullException("sender"); // Do not localize
			
			ShowAutoContainer(EffectiveEcoSpace(sender), CurrencyManagerHandle.CurrentElement(sender as Control), readOnlyComponents[sender] != null);
		}
		#endregion
		#region EcoSpace related
		private EcoSpace m_EcoSpace;
		/// <summary>
		/// Used as a fall back alternative to locate an EcoSpace. First the datasource of the extendee is followed to the root.
		/// </summary>
		[Browsable(false)]
		[DesignerSerializationVisibility(DesignerSerializationVisibility.Hidden)]
		public EcoSpace EcoSpace
		{
			get { return m_EcoSpace; }
			set { m_EcoSpace = value; }
		}
		#endregion

	}
}
